clear all
set matsize 10000
set maxvar 30000
set memory 20000m


*************************
* Define some Mata functions we'll use later
*************************
capture mata mata drop MakeAConstraint(nvars)
mata:
void MakeAConstraint(nvars)
{
	/*generate identity matrix of the right size*/
	A = I(nvars)
	
	/*fill lower triangle with dots */
	for (j=1 ; j<=nvars ; j++) {
		for (k=1 ; k<j ; k++) {
			A[j,k] = .
		}
	}
	
	st_matrix("A", A) 
}
end

capture mata mata drop MakeBConstraint(nvars)
mata:
void MakeBConstraint(nvars)
{
	/*generate zero matrix of the right size*/
	B = J(nvars,nvars,0)
	
	/*fill in the diagonal with dots*/
	for (j=1 ; j<=nvars ; j++) {
		B[j,j] = .
	}
	
	st_matrix("B", B) 
}
end


capture mata mata drop ComputeSimSeries(z1, z2)
mata:
void ComputeSimSeries(z1, z2)
{
	Ahat    = st_matrix("Ahat")
	Bhat    = st_matrix("Bhat")
	A0      = st_matrix("A0")
	A1      = st_matrix("A1")
	A2      = st_matrix("A2")
	Y       = st_matrix("Y")
	e_struc = st_matrix("e_struc")
	e_red   = st_matrix("e_red")
	
	/*loop through each row of Y*/
	printf("Computing structural shocks \n")
	nrows = rows(Y)
	ncols = cols(Y)
	for (t=3 ; t<=nrows ; t++) {
		printf("we're now on row number: %9.0g \n", t)
		
		/*compute e_struc for this observation*/
		Yt   = Y[t,1::ncols]'
		Yt_1 = Y[t-1,1::ncols]'
		Yt_2 = Y[t-2,1::ncols]'
		
		printf("compute e_struct \n")
		e_struct = luinv(Bhat)*Ahat*(Yt-A0-A1*Yt_1-A2*Yt_2)
		printf("compute e_redt")
		e_redt = (Yt-A0-A1*Yt_1-A2*Yt_2)'
		
		printf("append to e_struc \n")
		e_struc = e_struc\e_struct'
		printf("append to e_red \n")
		e_red   = e_red\e_redt
		
		Yt   = -999
		Yt_1 = -999
		Yt_2 = -999
	}
	
	/*make column of zeroes*/
	zeroes = J(nrows,1,0)
	
	/*[Step 4] make adjusted e_struc, which is the structural shocks when we set the policy shocks to zero. Set columns z1 and z2 to zero*/
	printf("Doing step 4 \n")
	e_struc_adj = e_struc
	e_struc_adj
	nrows
	z1
	z2
	e_struc_adj[1::nrows,z1] = zeroes
	e_struc_adj[1::nrows,z2] = zeroes
	
	/*[Step 5] compute the adjusted reduced-form shocks. Note that it still has 175 obs, even though two of them are missing.*/
	printf("Doing step 5 \n")
	e_red_adj = (luinv(Ahat)*Bhat*e_struc_adj')'
	
	/*[Step 6] Start with the first two obs of Y, and iterate forward using the adjusted reduced-form shocks*/
	printf("Doing step 6 \n")
	nrows = rows(Y)
	ncols = cols(Y)
	Y_adj = J(nrows,ncols,.)
	Y_adj[1::2,1::ncols] = Y[1::2,1::ncols]
	for (t=3 ; t<=nrows ; t++) {
		printf("we're now on row number: %9.0g \n", t)
		
		/*fill in the simulated values for observation t*/
		Yt_1 = Y_adj[t-1,1::ncols]'
		Yt_2 = Y_adj[t-2,1::ncols]'
		Y_adj[t,1::ncols] = (A0 + A1*Yt_1 + A2*Yt_2 + e_red_adj[t,1::ncols]')'
	}
	
	st_matrix("Y_adj", Y_adj) 
	st_matrix("e_struc", e_struc)
	st_matrix("e_red", e_red)
}
end




**************************
* Define the main function for running VARs
**************************
capture program drop RunVAR
program define RunVAR

	*Set Parameters
	*****************
	*specify the name for this set of results
	local FolderName `1'
	
	*set to data file that contains the policy variables, and the data file that contains the factors. Note: unless the factor file name says "NotSeparate", then the file contains factors computed using the activity inputs and the price inputs separated.
	local Factors    = "`2'"
	local PolicyVars = "../../(1) Data prep/clean data/transformed data (MOM CNYadj BW36 20001_201310).dta"
	
	*Set the number of lags desired in the VAR
	local nlags = 2
	
	*Define locals that represent the VAR specifications that we consider
	local Spec1 EA PR IntRt20d
	local Spec2 EA PR ReqResRat
	local Spec3 EA PR M2
	local Spec4 EA PR Lending
	local Spec5 EA PR GovtExp
	local Spec6 EA PR M2 ReqResRat IntRt20d
	local Spec7 EA PR Lending ReqResRat IntRt20d
	local Spec8 EA PR GovtExp ReqResRat IntRt20d
	local Spec8a EA PR ReqResRat GovtExp IntRt20d
	local Spec8b EA PR ReqResRat IntRt20d GovtExp
	local Spec8c EA GovtExp PR ReqResRat IntRt20d
	local Spec8d GovtExp EA PR ReqResRat IntRt20d

	local Spec11 EA EA2 PR IntRt20d
	local Spec12 EA EA2 PR ReqResRat
	local Spec13 EA EA2 PR M2
	local Spec14 EA EA2 PR Lending
	local Spec15 EA EA2 PR GovtExp
	local Spec16 EA EA2 PR M2 ReqResRat IntRt20d
	local Spec17 EA EA2 PR Lending ReqResRat IntRt20d
	
	local Spec31 IP_US EA PR IntRt20d
	local Spec32 IP_US EA PR ReqResRat
	local Spec33 IP_US EA PR M2
	local Spec34 IP_US EA PR Lending
	local Spec35 IP_US EA PR GovtExp
	local Spec36 IP_US EA PR M2 ReqResRat IntRt20d
	local Spec37 IP_US EA PR Lending ReqResRat IntRt20d
	local Spec38 IP_US EA PR GovtExp ReqResRat IntRt20d

	local Spec41 Commodity EA PR IntRt20d
	local Spec42 Commodity EA PR ReqResRat
	local Spec43 Commodity EA PR M2
	local Spec44 Commodity EA PR Lending
	local Spec45 Commodity EA PR GovtExp
	local Spec46 Commodity EA PR M2 ReqResRat IntRt20d
	local Spec47 Commodity EA PR Lending ReqResRat IntRt20d
	local Spec48 Commodity EA PR GovtExp ReqResRat IntRt20d

	local Spec56 IP_US Commodity EA PR M2 ReqResRat IntRt20d
	local Spec57 IP_US Commodity EA PR Lending ReqResRat IntRt20d
	local Spec58 IP_US Commodity EA PR GovtExp ReqResRat IntRt20d
	
	*set to the specifications we want to run, (e.g. you can set it to "1 2 3 4 5" to run specifications 1 through 5 listed above)
	local SpecsToRun `3'
	
	*set to the position number of the variables whose structural shocks we want to not include in the simulated series. If you only want to select one variable, set both locals to the same number
	local ZeroVar1 = 3
	local ZeroVar2 = 3

	*set to "DontShowAllIRFs" to not show the complete set of IRFs in the combined graph
	local DontShowAllIRFs = "`4'"
	if "`DontShowAllIRFs'"=="DontShowAllIRFs"{
		*set to 1 to only show the responses of EA1 and PR1
		local OnlyShowEAPRResponse = 1
		*set to 1 to not show 
		local DontShowEAPRImpulse = 1
	}
		
	*set to number of rows and columns in the combined graph. This depends on the specification you run and on which impulse responses you want to show.
	local nrows = `5'
	local ncols = `6'

	*set to 1 to graph the structural policy shock
	local GraphEStruc = 0

	*set to 1 to graph fitted values
	local GraphFitted = "`7'"

	*Set to the confidence level desired for IRFs
	local CI 68

	*create local to hold names of all the graphs (the local should be set to "")
	local AllIRFs = ""

	*set to 1 to also export graphs of the individual impulse response functions
	local ShowIndividualImpRes = 0

	*set to 1 to also export the data used in the VAR
	local ExportVARData = 0
	
	*set to "NotSeparate" if the factor inputs were not separated into EA and PR factor inputs. This means that the factors are named EA1 and EA2, where EA1 is what we'll treat as the EA factor and EA2 is what we'll treat as the price factor
	local NotSeparate = "`8'"

	*make folder to output all results (if this directory doesn't already exist)
	capture mkdir "../output/`FolderName'"
	
	

	
	*loop through each specification
	***************************************
	foreach i in `SpecsToRun' {

		disp "We're now running specification: `i'"

		***************************
		*Import data
		***************************
		*import estimated factors
		import excel "`Factors'", firstrow sheet("Factors") clear
		drop if EA1==.
		rename A dated
		gen date = mofd(dated)
		format date %tm
		drop dated
		
		*rename EA2 to be PR1 if the factor inputs were not separated into EA and PR factor inputs
		if "`NotSeparate'"=="NotSeparate" {
			rename EA2 PR1
			
			*multiply PR1 by -1 to get the sign comparable to the PR1 factor we get when we separate
			gen tmp = -1*PR1
			drop PR1
			rename tmp PR1
		}
		
		*merge with policy variables
		merge 1:1 date using "`PolicyVars'"
			*1999m1-m3 don't match, since the factors are missing those obs
			*also, everything after 2013m10 don't match, since those are in the future
		drop if _merge!=3
		drop _merge

		*rename variables so they have shorter names
		rename EA1                  EA
		rename PR1                  PR
		rename MoneySupplyM2        M2
		rename GovtExpenditure      GovtExp
		rename RequiredReserveRatio ReqResRat
		rename BenchmarkRate20d     IntRt20d

		* Normalize all variables to mean 0 variance 1
		ds date year month, not
		foreach v in `r(varlist)'{
			sum `v'
			replace `v' = (`v'-`r(mean)')/`r(sd)'
		}
		
		* Export data
		if `ExportVARData'==1 {
			export excel using ../output/`FolderName'/VARdata.xlsx, firstrow(var) sheet("data") sheetreplace datestring("%tm")
		}


		***************
		*Run VARs
		***************
		*[Step 1] Compute the estimated "A" and "B" matrices
		disp "compute mapping from reduced form shocks to structural shocks"
		local nvars = wordcount("`Spec`i''")
		
		*create the A and B constraint matrices for recursive ordering
		mata: MakeAConstraint(`nvars')
		mata: MakeBConstraint(`nvars')
		
		svar `Spec`i'', aeq(A) beq(B) lags(1/`nlags')
		matrix Ahat = e(A)
		matrix Bhat = e(B)
		
		*[Step 2] compute the coefficient matricies in reduced form VAR
		disp "Step 2: compute A0, A1, and A2"

		*run the var
		var `Spec`i'', lags(1/`nlags') 
		
		matrix b = e(b)

		if `nvars'==3{
			forvalues j = 1/21{
				local b`j' = b[1,`j']
			}
			matrix A0 = (`b7' \ `b14' \ `b21')
			matrix A1 = (`b1', `b3', `b5' \ `b8', `b10', `b12' \ `b15', `b17', `b19')
			matrix A2 = (`b2', `b4', `b6' \ `b9', `b11', `b13' \ `b16', `b18', `b20')
		}
		
		if `nvars'==4{
			forvalues j = 1/36{
				local b`j' = b[1,`j']
			}
			matrix A0 = (`b9' \ `b18' \ `b27' \ `b36')
			matrix A1 = (`b1', `b3', `b5', `b7' \ `b10', `b12', `b14', `b16' \ `b19', `b21', `b23', `b25' \ `b28', `b30', `b32', `b34')
			matrix A2 = (`b2', `b4', `b6', `b8' \ `b11', `b13', `b15', `b17' \ `b20', `b22', `b24', `b26' \ `b29', `b31', `b33', `b35')
		}
			
		if `nvars'==5{
			forvalues j = 1/55{
				local b`j' = b[1,`j']
			}
			matrix A0 = (`b11' \ `b22' \ `b33' \ `b44' \ `b55' )
			matrix A1 = (`b1', `b3', `b5', `b7', `b9' \  `b12', `b14', `b16', `b18', `b20' \ `b23', `b25', `b27', `b29', `b31' \ `b34', `b36', `b38', `b40', `b42' \ `b45', `b47', `b49', `b51', `b53')
			matrix A2 = (`b2', `b4', `b6', `b8', `b10' \ `b13', `b15', `b17', `b19', `b21' \ `b24', `b26', `b28', `b30', `b32' \ `b35', `b37', `b39', `b41', `b43' \ `b46', `b48', `b50', `b52', `b54')
		}

		if `nvars'==6{
			forvalues j = 1/78{
				local b`j' = b[1,`j']
			}
			matrix A0 = (`b13' \ `b26' \ `b39' \ `b52' \ `b65' \ `b78' )
			matrix A1 = (`b1', `b3', `b5', `b7', `b9',  `b11' \  `b14', `b16', `b18', `b20', `b22', `b24' \ `b27', `b29', `b31', `b33', `b35', `b37' \ `b40', `b42', `b44', `b46', `b48', `b50' \ `b53', `b55', `b57', `b59', `b61', `b63' \ `b66', `b68', `b70', `b72', `b74', `b76')
			matrix A2 = (`b2', `b4', `b6', `b8', `b10', `b12' \  `b15', `b17', `b19', `b21', `b23', `b25' \ `b28', `b30', `b32', `b34', `b36', `b38' \ `b41', `b43', `b45', `b47', `b49', `b51' \ `b54', `b56', `b58', `b60', `b62', `b64' \ `b67', `b69', `b71', `b73', `b75', `b77')
		}
		
		if `nvars'==7{
			forvalues j = 1/105{
				local b`j' = b[1,`j']
			}
			
			matrix A0 = (`b15' \ `b30' \ `b45' \ `b60' \ `b75' \ `b90' \ `b105' )
			matrix A1 = (`b1', `b3', `b5', `b7', `b9',  `b11', `b13' \  `b16', `b18', `b20', `b22', `b24', `b26', `b28' \ `b31', `b33', `b35', `b37', `b39', `b41', `b43' \ `b46', `b48', `b50', `b52', `b54', `b56', `b58' \ `b61', `b63', `b65', `b67', `b69', `b71', `b73' \ `b76', `b78', `b80', `b82', `b84', `b86', `b88' \ `b91', `b93', `b95', `b97', `b99',  `b101', `b103' )
			matrix A2 = (`b2', `b4', `b6', `b8', `b10', `b12', `b14' \  `b17', `b19', `b21', `b23', `b25', `b27', `b29' \ `b32', `b34', `b36', `b38', `b40', `b42', `b44' \ `b47', `b49', `b51', `b53', `b55', `b57', `b59' \ `b62', `b64', `b66', `b68', `b70', `b72', `b74' \ `b77', `b79', `b81', `b83', `b85', `b87', `b89' \ `b92', `b94', `b96', `b98', `b100', `b102', `b104' )
		}

		*[Step 3] compute the estimated structural shocks
		*convert the three variables into matrices
		mkmat `Spec`i''
		local YVars = subinstr("`Spec`i''"," ",",",.)
		matrix Y = `YVars'

		*Rows of Y correspond to observations of the variables at a particular month. So, for a given t, Yt is a single row of the matrix Y, inverted to be a column vector.
		*start the matrix off with two obs of missing values, since we'll only have structural shocks for observatiosn 3 onward due to the two lags
		if `nvars'==3{
			matrix e_struc = (.,.,.\.,.,.)
			matrix e_red   = (.,.,.\.,.,.)
		}
		if `nvars'==4{
			matrix e_struc = (.,.,.,.\.,.,.,.)
			matrix e_red   = (.,.,.,.\.,.,.,.)
		}
		if `nvars'==5{
			matrix e_struc = (.,.,.,.,.\.,.,.,.,.)
			matrix e_red   = (.,.,.,.,.\.,.,.,.,.)
		}
		if `nvars'==6{
			matrix e_struc = (.,.,.,.,.,.\.,.,.,.,.,.)
			matrix e_red   = (.,.,.,.,.,.\.,.,.,.,.,.)
		}
		if `nvars'==7{
			matrix e_struc = (.,.,.,.,.,.,.\.,.,.,.,.,.,.)
			matrix e_red   = (.,.,.,.,.,.,.\.,.,.,.,.,.,.)
		}
		
		*create the simulated series, with and without the policy shock(s). The function will spit out the matrices Y and Y_adj, which hold the variables of interest with and without the policy shocks.
		disp "running the ComputeSimSeries function..."
		mata: ComputeSimSeries(`ZeroVar1',`ZeroVar2')
		
		*now convert matrices Y, Y_adj, e_struc, and e_red back into variables
		svmat Y
		svmat Y_adj
		svmat e_struc
		svmat e_red
		
		*graph the difference between the Y and Y_adj variables
		if "`GraphFitted'"=="GraphFitted" {
			local nvars = wordcount("`Spec`i''")
			forvalues j = 1/`nvars' {
				
				disp "taking 12-month MA of unadjusted and adjusted series..."
				tsset date
				*take 12-month moving averages of Y and Y_adj variables
				foreach v in Y`j' Y_adj`j'{
					gen `v'2 = (`v' + L1.`v' + L2.`v' + L3.`v' + L4.`v' + L5.`v' + L6.`v' + L7.`v' + L8.`v' + L9.`v' + L10.`v' + L11.`v')/12
					drop `v'
					rename `v'2 `v'
				}
				
				capture drop policy_contribution`j'
				gen policy_contribution`j' = Y`j'-Y_adj`j'

				twoway line Y_adj`j' Y`j' date, xlabel(#6, labsize(medium) ) ylabel(, labsize(medium)) legend(off) lpattern(solid dash) scheme(s1mono)
				graph export "../output/`FolderName'/Spec(`Spec`i'') Off(`ZeroVar1' `ZeroVar2') Var(`j').eps", as(eps) preview(on) mag(200) orientation(landscape) fontface("Times New Roman") replace
				
				capture drop policy_contribution`j'
				capture drop Y_adj`j'
				capture drop Y`j'
			}
		}
		
		if `GraphEStruc'==1{
			*graph the 3rd column of e_struc, which is the series of structural policy shocks
			disp "graphing the structural policy shocks..."
			twoway line e_struc3 date, title("Spec(`Spec`i'') Policy innovations") xlabel(, labsize(huge)) ylabel(, labsize(huge))
			graph export "../output/graphs/sim/Spec(`Spec`i'') Policy innovations.png", as(png) replace
		}
		
		*graph IRFs
		local ImpulseVars = "`Spec`i''"
		if "`DontShowEAPRImpulse'"=="1"{
			local ImpulseVars = subinstr("`ImpulseVars'","EA","",.)
			local ImpulseVars = subinstr("`ImpulseVars'","PR","",.)
		}
		
		local ResponseVars = "`Spec`i''"
		if "`OnlyShowEAPRResponse'"=="1"{
			local ResponseVars = "EA PR"
		}
		
		irf create irf`i', step(20) set("../data/temp_files/set.irf", replace) order(`spec`i'')
		
		foreach impvar in `ImpulseVars' {
		foreach resvar in `ResponseVars'{
			
			** If you want the additional functionality of the "irf graph" function, you can replace the preserve-restore section below with the following three lines
			*irf graph coirf, byopts(yrescale title("") legend(off) note("")) level(`CI') impulse(`impvar') response(`resvar') title("Response of `resvar' to `impvar'",size(medium)) subtitle("") ytitle("") xtitle("period", size(small)) xlabel(, labsize(medium)) ylabel(, labsize(medium)) graphregion(margin(tiny)) scale(0.5)
			*graph export "../output/`FolderName'/`Spec`i'' (`impvar' on `resvar').png", as(png) replace
			*graph save "`impvar'_`resvar'.gph", replace		
			
			preserve
				use ../data/temp_files/set.irf, clear
				keep if impulse=="`impvar'" & response=="`resvar'"
				sort step
				
				gen high = coirf+stdcoirf
				gen low  = coirf-stdcoirf
				
				*Use the variable names in the graph title, unless it's IntRt20d. We want a more intuitive name for that variable.
				if "`impvar'"=="IntRt20d"{
					local impvarlab ShortRate
				}
				else{
					local impvarlab `impvar'
				}
				if "`resvar'"=="IntRt20d"{
					local resvarlab ShortRate
				}					
				else{
					local resvarlab `resvar'
				}
				twoway (rarea high low step, fcolor(gs14) lcolor(gs16)) (line coirf step, lcolor(black)), title("Response of" "`resvarlab' to `impvarlab'", size(large)) subtitle("") ytitle("") xtitle("period", size(medlarge)) xlabel(, labsize(medlarge)) ylabel(, labsize(medlarge)) ylabel(#3) graphregion(margin(tiny)) scale(0.5) legend(off)
				
				if `ShowIndividualImpRes'==1{
					graph export "../output/`FolderName'/`Spec`i'' (`impvar' on `resvar').png", as(png) replace
				}
				graph save "../data/temp_files/`impvar'_`resvar'.gph", replace				
			restore
			
			local AllIRFs = `"  `AllIRFs'   "../data/temp_files/`impvar'_`resvar'.gph" "'
		}
		}
		
		matrix drop Y Y_adj
		matrix drop e_red e_struc `Spec`i'' A2 A1 A0 b Bhat Ahat B A
		
	*end loop through all specifications
	}

	
	*create a combined graph showing all IRF graphs
	*************************************************
	disp "now making combined graph..."
	
	*determine if you want the graph in landscape or portrait, and specify graph's width
	if `ncols'>=5{
		local LandOrPort = "landscape"
		local XSize = 5
	}
	if `ncols'<5{
		local LandOrPort = "portrait"
		local XSize = 4
	}
	
	*make and export graph
	graph combine `AllIRFs', rows(`nrows') cols(`ncols') altshrink commonscheme scheme(s1mono) xcommon iscale(4) xsize(`XSize') ysize(4)
	graph export "../output/`FolderName'/AllIRFs.eps", as(eps) preview(on) mag(200) orientation(`LandOrPort') fontface("Times New Roman") replace

end



*make figures
RunVAR Figure2  "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "1 2 3 4 5"  DontShowAllIRFs 5 2  ""          Separate
RunVAR Figure3  "../data/factors_NarrowMOM (CNYadj BW36 20001_201310).xls"              "1 2 3 4 5"  DontShowAllIRFs 5 2  ""          Separate
RunVAR Figure4  "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "6"          ShowAllIRFs     5 5  ""          Separate
RunVAR Figure5  "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "7"          ShowAllIRFs     5 5  ""          Separate
RunVAR Figure6a "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "1"          ShowAllIRFs     3 3  GraphFitted Separate
RunVAR Figure6b "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "2"          ShowAllIRFs     3 3  GraphFitted Separate
RunVAR Figure7  "../data/factors_BroadMOM (CNYadj BW36 20001_201310).xls"               "56"         ShowAllIRFs     7 7  ""          Separate
RunVAR FigureA1 "../data/factors_BroadMOM (CNYadj BW36 20001_201310) NotSeparate.xls"   "6"          ShowAllIRFs     5 5  ""          NotSeparate








